// engine.js
import { rules } from "./rules.js";

// Entropy calculation function
function calculateEntropy(str) {
  const map = {};
  for (const char of str) {
    map[char] = (map[char] || 0) + 1;
  }

  let entropy = 0;
  for (const char in map) {
    const p = map[char] / str.length;
    entropy -= p * Math.log2(p);
  }

  return entropy;
}

export function analyzeURL(url) {
  let score = 0;
  const reasons = [];

  // Extract hostname
  const hostname = url.replace(/^https?:\/\//, "").split("/")[0];

  // 1. Length check
  if (url.length > 75) {
    score += 10;
    reasons.push("URL is unusually long");
  }

  // 2. IP address usage
  if (/https?:\/\/\d+\.\d+\.\d+\.\d+/.test(url)) {
    score += 25;
    reasons.push("Uses raw IP address");
  }

  // 3. Suspicious keywords
  rules.suspiciousKeywords.forEach(word => {
    if (url.toLowerCase().includes(word)) {
      score += 5;
      reasons.push(`Contains suspicious keyword: "${word}"`);
    }
  });

  // 4. Dangerous TLDs
  rules.dangerousTlds.forEach(tld => {
    if (url.endsWith(tld)) {
      score += 20;
      reasons.push(`Uses risky TLD: ${tld}`);
    }
  });

  // 5. Subdomain depth
  const subdomains = hostname.split(".").length - 2;
  if (subdomains > rules.maxSubdomains) {
    score += 15;
    reasons.push("Excessive subdomains detected");
  }

  // 6. Entropy analysis
  const domain = hostname.split(".")[0];
  const entropy = calculateEntropy(domain);
  if (entropy > rules.entropyThreshold) {
    score += 20;
    reasons.push(`High domain entropy detected (${entropy.toFixed(2)})`);
  }

  // 7. Homoglyph detection
  Object.keys(rules.homoglyphs).forEach(fake => {
    if (hostname.includes(fake)) {
      score += 15;
      reasons.push(
        `Possible homoglyph attack: "${fake}" used instead of "${rules.homoglyphs[fake]}"`
      );
    }
  });

  // 8. Final verdict with explanation
  let verdict = "SAFE";
  let explanation = "No significant risk detected.";

  if (score >= 60) {
    verdict = "DANGEROUS";
    explanation = "High-risk URL. Likely phishing or scam.";
  } else if (score >= 30) {
    verdict = "WARNING";
    explanation = "Suspicious patterns found. Review the reasons.";
  }

  return { verdict, score, reasons, explanation };
}
